import { io } from 'socket.io-client'
import { onUnmounted, ref } from 'vue'

export const useVoiceRoom = (roomId, currentUserId, roleItem, campaignUsers) => {
    const socket = io('https://mahovei-cdn.ir', {
        transports: ['websocket'],
        secure: true,
    })

    const peers = {}
    const localStream = ref(null)
    const role = ref(roleItem === 'host' ? 'speaker' : 'listener')
    const myId = ref(null)
    const users = ref({})
    const pendingCandidates = {} // { userId: [candidate1, candidate2, ...] }

    const { $sweetalert } = useNuxtApp()

    // --- iOS detection & unlock ---
    // --- iOS detection & unlock ---
    const isIOS = () =>
        /iPad|iPhone|iPod/.test(navigator.userAgent) ||
        (navigator.platform === 'MacIntel' && navigator.maxTouchPoints > 1)

    const ensurePlaybackUnlocked = (() => {
        let unlocked = false
        let ctx = null
        const AC =
            (typeof window !== 'undefined') &&
            (window.AudioContext || window.webkitAudioContext)

        return async () => {
            if (unlocked) return true
            try {
                if (AC) {
                    if (!ctx) ctx = new AC()
                    if (ctx.state === 'suspended') await ctx.resume()
                    // ترفند مطمئن: یک صدای خیلی کوتاه و بی‌صدا
                    const osc = ctx.createOscillator()
                    const gain = ctx.createGain()
                    gain.gain.value = 0.0001
                    osc.connect(gain).connect(ctx.destination)
                    osc.start()
                    osc.stop(ctx.currentTime + 0.01)
                }
                unlocked = true
                return true
            } catch (e) {
                console.warn('AudioContext unlock failed', e)
                return false
            }
        }
    })()

    const showiOSUnlockButton = (onUnlock) => {
        if (document.querySelector('#ios-audio-unlock')) return
        const btn = document.createElement('button')
        btn.id = 'ios-audio-unlock'
        btn.innerText = 'فعال‌سازی صدا'
        btn.onclick = async () => {
            try {
                await ensurePlaybackUnlocked()
                await onUnlock?.();
                changeMicStatus(false);
            } finally {
                btn.remove()
            }
        }

        const roomBox = document.getElementById('room-box');
        roomBox.appendChild(btn)
    }

    const createPeer = (id) => {
        const peer = new RTCPeerConnection({
            iceServers: [
                { urls: 'stun:stun.l.google.com:19302' },
                // 🔒 TURN خودتان را جایگزین کنید
                {
                    urls: ['turn:YOUR_TURN_DOMAIN:3478', 'turns:YOUR_TURN_DOMAIN:5349'],
                    username: 'YOUR_TURN_USERNAME',
                    credential: 'YOUR_TURN_PASSWORD'
                }
            ]
        })

        // Logs
        peer.oniceconnectionstatechange = () => {
            console.log('🧊 iceConnectionState:', peer.iceConnectionState)
        }
        peer.onconnectionstatechange = () => {
            console.log('🔗 connectionState:', peer.connectionState)
        }
        peer.onsignalingstatechange = () => {
            console.log('📶 signalingState:', peer.signalingState)
        }
        peer.onnegotiationneeded = () => {
            console.log('🧩 negotiationneeded')
        }

        peer.onicecandidate = (e) => {
            if (e.candidate) {
                const targetSocketId = users.value[id]?.socketId
                if (targetSocketId) {
                    socket.emit('ice-candidate', { target: targetSocketId, candidate: e.candidate })
                }
            }
        }

        // util: create/reuse audio element safely
        const ensureAudioEl = async (stream) => {
            const sid = stream.id
            let audio = document.querySelector(`audio[data-stream-id="${sid}"]`)
            if (!audio) {
                audio = document.createElement('audio')
                audio.dataset.streamId = sid
                audio.autoplay = true
                audio.controls = true
                audio.playsInline = true
                audio.muted = false
                audio.volume = 1.0
                // از display:none استفاده نکنید (مشکل iOS)
                audio.style.position = 'absolute'
                audio.style.left = '-9999px'
                audio.style.width = '0'
                audio.style.height = '0'
                audio.classList.add('hidden-audio', 'listener')
                document.body.appendChild(audio)
            }
            audio.srcObject = stream

            if ('setSinkId' in HTMLMediaElement.prototype) {
                try { await audio.setSinkId('default') } catch (e) { console.warn('setSinkId failed', e) }
            }
            return audio
        }

        peer.ontrack = async ({ streams, track }) => {
            const stream = streams?.[0]
            if (!stream) return

            console.log('📡 دریافت استریم از اسپیکر:', stream)
            const audio = await ensureAudioEl(stream)

            const tryPlay = async () => {
                try {
                    await ensurePlaybackUnlocked()
                    await audio.play()
                    console.log('🔊 Audio is playing')
                } catch (err) {
                    console.warn('📱 صدا در موبایل اجرا نشد، نیاز به تعامل کاربر دارد.', err)
                    const btn = document.createElement('button')
                    btn.innerText = 'پخش صدا'
                    btn.classList.add('manual-play-button')
                    btn.style.position = 'fixed'
                    btn.style.zIndex = '99999'
                    btn.style.bottom = '16px'
                    btn.style.left = '16px'
                    btn.onclick = async () => {
                        await ensurePlaybackUnlocked()
                        await audio.play().then(() => btn.remove()).catch(console.warn)
                    }
                    document.body.appendChild(btn)
                }
            }

            if (track?.muted) {
                console.warn('🔇 remote track muted; waiting for onunmute…')
                track.onunmute = () => {
                    console.log('🔊 remote track unmuted')
                    tryPlay()
                }
            } else {
                tryPlay()
            }

            setTimeout(async () => {
                if (audio.paused) {
                    console.warn('⏱ Still paused after 1s, trying again…')
                    await tryPlay()
                }
            }, 1000)
        }

        return peer
    }

    const startSpeaker = async () => {
        if (role.value !== 'speaker') return

        localStream.value = await navigator.mediaDevices.getUserMedia({
            audio: { echoCancellation: true, noiseSuppression: true, autoGainControl: true }
        })
        const track = localStream.value.getAudioTracks()?.[0]
        console.log('🎤 Mic track info:', track && {
            enabled: track.enabled,
            muted: track.muted,
            readyState: track.readyState
        })

        const audio = document.createElement('audio')
        audio.srcObject = localStream.value
        audio.muted = true
        audio.autoplay = true
        audio.controls = true
        audio.classList.add('hidden-audio', 'speaker')
        audio.style.position = 'absolute'
        audio.style.left = '-9999px'
        document.body.appendChild(audio)
    }

    // ارسال Offer به همه‌ی غیرخود
    const createOffersToAll = async () => {
        for (const [userId, user] of Object.entries(users.value)) {
            if (userId === myId.value) continue
            const peer = createPeer(userId)
            peers[userId] = peer
            localStream.value.getTracks().forEach(t => peer.addTrack(t, localStream.value))
            const offer = await peer.createOffer({ offerToReceiveAudio: 1 })
            await peer.setLocalDescription(offer)
            socket.emit('offer', { target: user.socketId, sdp: offer, caller: myId.value })
        }
    }

    const setupSocketListeners = () => {
        socket.on('offer', async ({ caller, sdp }) => {
            const peer = createPeer(caller)
            peers[caller] = peer

            // برای Safari/iOS: تضمین خطوط صوتی recvonly قبل از setRemoteDescription
            try {
                const hasAudioRecv = peer.getTransceivers?.().some(t => t.receiver?.track?.kind === 'audio')
                if (!hasAudioRecv) {
                    peer.addTransceiver('audio', { direction: 'recvonly' })
                }
            } catch (e) {
                // قدیمی‌ها ممکن است getTransceivers نداشته باشند
            }

            await peer.setRemoteDescription(new RTCSessionDescription(sdp))

            if (pendingCandidates[caller]) {
                for (const c of pendingCandidates[caller]) {
                    try { await peer.addIceCandidate(c) } catch (err) {
                        console.error('🔥 Error adding queued ICE candidate:', err)
                    }
                }
                delete pendingCandidates[caller]
            }

            const answer = await peer.createAnswer()
            await peer.setLocalDescription(answer)

            const callerSocketId = users.value[caller]?.socketId
            if (callerSocketId) {
                socket.emit('answer', { target: callerSocketId, sdp: answer, caller: myId.value })
            }
        })

        socket.on('answer', async ({ caller, sdp }) => {
            const peer = peers[caller]
            if (!peer) return

            try {
                if (peer.signalingState !== 'have-local-offer') {
                    console.warn('⛔ Cannot apply answer; signalingState is', peer.signalingState, 'for caller', caller)
                    return
                }

                await peer.setRemoteDescription(new RTCSessionDescription(sdp))

                if (pendingCandidates[caller]) {
                    for (const c of pendingCandidates[caller]) {
                        try { await peer.addIceCandidate(c) } catch (err) {
                            console.error('🔥 Error adding queued ICE candidate:', err)
                        }
                    }
                    delete pendingCandidates[caller]
                }

                console.log('✅ Answer applied successfully for', caller)
            } catch (error) {
                console.error('❌ Failed to set remote answer:', error)
            }
        })

        socket.on('ice-candidate', async ({ from, candidate }) => {
            const fromUserId = Object.keys(users.value).find(uid => users.value[uid].socketId === from)
            const peer = peers[fromUserId]
            if (!peer || !candidate) return

            const rtcCandidate = new RTCIceCandidate(candidate)

            if (peer.remoteDescription && peer.remoteDescription.type) {
                try {
                    await peer.addIceCandidate(rtcCandidate)
                } catch (err) {
                    console.error('🔥 Error adding candidate AFTER description:', err)
                }
            } else {
                if (!pendingCandidates[fromUserId]) pendingCandidates[fromUserId] = []
                pendingCandidates[fromUserId].push(rtcCandidate)
                console.warn('📥 ICE candidate queued for', fromUserId)
            }
        })

        socket.on('role-changed', async (newRole) => {
            role.value = newRole
            if (newRole === 'speaker') {
                if (isIOS()) {
                    showiOSUnlockButton(async () => {
                        await startSpeaker()
                        await createOffersToAll()
                    })
                } else {
                    await startSpeaker()
                    await createOffersToAll()
                }
            } else {
                if (localStream.value) {
                    localStream.value.getTracks().forEach(track => track.stop())
                    localStream.value = null
                }
            }
        })

        socket.on('users-update', async (updatedUsers) => {
            const newUsers = Object.keys(updatedUsers).filter(id => !(id in users.value) && id !== myId.value)
            users.value = updatedUsers

            console.log(users.value)

            if (role.value === 'speaker' && localStream.value && newUsers.length > 0) {
                for (const userId of newUsers) {
                    const user = updatedUsers[userId]
                    const peer = createPeer(userId)
                    peers[userId] = peer
                    localStream.value.getTracks().forEach(track => peer.addTrack(track, localStream.value))
                    const offer = await peer.createOffer({ offerToReceiveAudio: 1 })
                    await peer.setLocalDescription(offer)
                    socket.emit('offer', { target: user.socketId, sdp: offer, caller: myId.value })
                }
            }
        })

        socket.on('hand-raised', ({ from }) => {
            const requester = users.value[from]
            if (requester) {
                $sweetalert.confirm(() => {
                    socket.emit('promote-to-speaker', { room: roomId, targetId: from })
                }, `کاربر ${requester.name} درخواست صحبت دارد. قبول می‌کنید؟`)
            } else {
                console.warn('❌ کاربر درخواست‌دهنده یافت نشد:', from)
            }
        })
    }

    const joinRoom = () => {
        socket.emit('join', {
            room: roomId,
            role: role.value,
            userId: currentUserId,
            roleName: roleItem,
            micState: false,
            image: campaignUsers.value.find(item => item.id === currentUserId)?.image,
            name: campaignUsers.value.find(item => item.id === currentUserId)?.name || 'بدون‌نام'
        })

        setupSocketListeners()

        socket.on('joined', async ({ id, role: assignedRole, users: joinedUsers }) => {
            myId.value = id
            role.value = assignedRole
            users.value = joinedUsers

            if (role.value === 'speaker') {
                if (isIOS()) {
                    showiOSUnlockButton(async () => {
                        await startSpeaker()
                        await createOffersToAll()
                    })
                } else {
                    await startSpeaker()
                    await createOffersToAll()
                }
            }

            if (role.value === 'listener') {
                socket.emit('listener-joined', { room: roomId, listenerId: myId.value })
            }
        })

        socket.on('resend-offer', async (listenerId) => {
            const user = Object.entries(users.value).find(([id, u]) => u.socketId === listenerId)
            if (!user) return

            const [userId, userInfo] = user

            const peer = createPeer(userId)
            peers[userId] = peer
            localStream.value?.getTracks().forEach(track => peer.addTrack(track, localStream.value))
            const offer = await peer.createOffer({ offerToReceiveAudio: 1 })
            await peer.setLocalDescription(offer)
            socket.emit('offer', { target: userInfo.socketId, sdp: offer, caller: myId.value })
        })

        socket.on('send-offer-to-listener', async ({ listenerId }) => {
            const user = Object.entries(users.value).find(([id, u]) => u.socketId === listenerId)
            if (!user) return

            const [userId, userInfo] = user

            const peer = createPeer(userId)
            peers[userId] = peer
            localStream.value?.getTracks().forEach(track => peer.addTrack(track, localStream.value))
            const offer = await peer.createOffer({ offerToReceiveAudio: 1 })
            await peer.setLocalDescription(offer)
            socket.emit('offer', { target: userInfo.socketId, sdp: offer, caller: myId.value })
        })

        socket.on('reaction', ({ userId, emoji }) => {
            const el = document.querySelector(`#reaction-${userId}`)
            if (el) {
                el.innerText = emoji
                el.classList.add('show')
                setTimeout(() => {
                    el.classList.remove('show')
                    el.innerText = ''
                }, 3000)
            } else {
                console.warn('❌ Reaction element not found for userId:', userId)
            }
        })

        socket.on('room-cleared', () => {
            for (const peer of Object.values(peers)) {
                peer.onicecandidate = null
                peer.ontrack = null
                peer.close()
            }

            if (localStream.value) {
                localStream.value.getTracks().forEach(track => track.stop())
                localStream.value = null
            }

            Object.keys(peers).forEach(id => delete peers[id])

            document.querySelectorAll('audio.hidden-audio').forEach(audio => {
                audio.pause()
                audio.srcObject = null
                audio.remove()
            })

            window.dispatchEvent(new CustomEvent('room-cleared-by-host'))

            socket.disconnect()
        })
    }

    const raiseHand = () => {
        socket.emit('raise-hand', roomId)
    }

    // ✅ fix: micEnabled یعنی «روشن باشد»
    const changeMicStatus = (micEnabled) => {
        if (localStream.value) {
            localStream.value.getAudioTracks().forEach(track => {
                track.enabled = !micEnabled;
                socket.emit('mic-change-state', {room: roomId,micState: !micEnabled})

                console.log('🎛️ mic track -> enabled:', track.enabled)
            });
        }
    };

    const clearRoom = () => {
        socket.emit('clear-room', roomId)
        document.querySelectorAll('audio.hidden-audio').forEach(audio => {
            audio.pause()
            audio.srcObject = null
            audio.remove()
        })
    }

    const leaveRoom = () => {
        socket.emit('leave-room', roomId)

        for (const peer of Object.values(peers)) {
            peer.onicecandidate = null
            peer.ontrack = null
            peer.close()
        }

        if (localStream.value) {
            localStream.value.getTracks().forEach(track => track.stop())
            localStream.value = null
        }

        Object.keys(peers).forEach(id => delete peers[id])

        document.querySelectorAll('audio.hidden-audio').forEach(audio => {
            audio.pause()
            audio.srcObject = null
            audio.remove()
        })

        socket.disconnect()
    }

    const muteListener = () => {
        if (roleItem === 'host') {
            socket.emit('force-listeners', roomId)
        }
    }

    const sendReaction = (emoji) => {
        socket.emit('send-reaction', { room: roomId, emoji })
    }

    return {
        joinRoom,
        raiseHand,
        leaveRoom,
        isIOS,
        role,
        users,
        clearRoom,
        muteListener,
        sendReaction,
        changeMicStatus,
        localStream
    }
}
