import { io } from 'socket.io-client'
import { onUnmounted, ref } from 'vue'

export const useVoiceRoom = (roomId, currentUserId, roleItem, campaignUsers) => {
    const socket = io('https://mahovei-cdn.ir', {
        transports: ['websocket'],
        secure: true,
    });

    const peers = {};
    const localStream = ref(null);
    const role = ref(roleItem === 'host' ? 'speaker' : 'listener');
    const myId = ref(null);
    const users = ref({});

    const { $sweetalert } = useNuxtApp();

    const createPeer = (id) => {
        const peer = new RTCPeerConnection({
            iceServers: [{ urls: 'stun:stun.l.google.com:19302' }]
        });

        peer.onicecandidate = (e) => {
            if (e.candidate) {
                const targetSocketId = users.value[id]?.socketId;
                if (targetSocketId) {
                    socket.emit('ice-candidate', { target: targetSocketId, candidate: e.candidate });
                }
            }
        };

        peer.ontrack = ({ streams }) => {
            console.log('📡 دریافت استریم از اسپیکر:', streams[0]);
            // if (localStream.value && streams[0].id === localStream.value.id) return;

            const existing = document.querySelector(`audio[data-stream-id="${streams[0].id}"]`);
            if (existing) return;

            const audio = document.createElement('audio');
            audio.srcObject = streams[0];
            audio.dataset.streamId = streams[0].id;
            audio.autoplay = true;
            audio.controls = true;
            audio.muted = false;
            audio.classList.add('hidden-audio');
            audio.classList.add('listener');
            document.body.appendChild(audio);

        };

        return peer;
    };


    const startSpeaker = async () => {
        if (role.value !== 'speaker') return;

        localStream.value = await navigator.mediaDevices.getUserMedia({ audio: true });

        const audio = document.createElement('audio');
        audio.srcObject = localStream.value;
        audio.muted = true;
        audio.autoplay = true;
        audio.controls = true;
        audio.classList.add('hidden-audio');
        audio.classList.add('speaker');
        document.body.appendChild(audio);
    };

    const setupSocketListeners = () => {
        socket.on('offer', async ({ caller, sdp }) => {
            const peer = createPeer(caller);
            peers[caller] = peer;

            await peer.setRemoteDescription(new RTCSessionDescription(sdp));
            const answer = await peer.createAnswer();
            await peer.setLocalDescription(answer);

            const callerSocketId = users.value[caller]?.socketId;
            if (callerSocketId) {
                socket.emit('answer', { target: callerSocketId, sdp: answer, caller: myId.value });
            }
        });

        socket.on('answer', async ({ caller, sdp }) => {
            const peer = peers[caller];
            if (!peer) return;

            try {
                if (peer.signalingState !== 'have-local-offer') {
                    console.warn('⛔ Cannot apply answer; signalingState is', peer.signalingState, 'for caller', caller);
                    return;
                }

                await peer.setRemoteDescription(new RTCSessionDescription(sdp));
                console.log('✅ Answer applied successfully for', caller);
            } catch (error) {
                console.error('❌ Failed to set remote answer:', error);
            }
        });


        socket.on('ice-candidate', async ({ from, candidate }) => {
            const fromUserId = Object.keys(users.value).find(uid => users.value[uid].socketId === from);
            const peer = peers[fromUserId];
            if (peer && candidate) {
                try {
                    await peer.addIceCandidate(new RTCIceCandidate(candidate));
                } catch (err) {
                    console.error('ICE Error:', err);
                }
            }
        });

        socket.on('role-changed', async (newRole) => {
            role.value = newRole;
            if (newRole === 'speaker') {
                await startSpeaker();

                for (const [userId, user] of Object.entries(users.value)) {
                    if (userId === myId.value) continue;
                    const peer = createPeer(userId);
                    peers[userId] = peer;

                    localStream.value.getTracks().forEach(track => {
                        peer.addTrack(track, localStream.value);
                    });

                    const offer = await peer.createOffer();
                    await peer.setLocalDescription(offer);

                    socket.emit('offer', {
                        target: user.socketId,
                        sdp: offer,
                        caller: myId.value
                    });
                }
            } else {
                if (localStream.value) {
                    localStream.value.getTracks().forEach(track => track.stop());
                    localStream.value = null;
                }
            }
        });

        socket.on('users-update', async (updatedUsers) => {
            const newUsers = Object.keys(updatedUsers).filter(id => !(id in users.value) && id !== myId.value);
            users.value = updatedUsers;

            if (role.value === 'speaker' && localStream.value && newUsers.length > 0) {
                for (const userId of newUsers) {
                    const user = updatedUsers[userId];
                    const peer = createPeer(userId);
                    peers[userId] = peer;

                    localStream.value.getTracks().forEach(track => {
                        peer.addTrack(track, localStream.value);
                    });

                    const offer = await peer.createOffer();
                    await peer.setLocalDescription(offer);

                    socket.emit('offer', {
                        target: user.socketId,
                        sdp: offer,
                        caller: myId.value
                    });
                }
            }
        });

        socket.on('hand-raised', ({ from }) => {
            const requester = users.value[from];
            if (requester) {
                $sweetalert.confirm(() => {
                    socket.emit('promote-to-speaker', { room: roomId, targetId: from });
                }, `کاربر ${requester.name} درخواست صحبت دارد. قبول می‌کنید؟`);
            } else {
                console.warn('❌ کاربر درخواست‌دهنده یافت نشد:', from);
            }
        });
    };


    const joinRoom = () => {
        socket.emit('join', {
            room: roomId,
            role: role.value,
            userId: currentUserId,
            roleName: roleItem,
            image: campaignUsers.value.find(item => item.id === currentUserId)?.image,
            name: campaignUsers.value.find(item => item.id === currentUserId)?.name || 'بدون‌نام'
        });

        setupSocketListeners();

        socket.on('joined', async ({ id, role: assignedRole, users: joinedUsers }) => {
            myId.value = id;
            role.value = assignedRole;
            users.value = joinedUsers;

            if (role.value === 'speaker') {
                await startSpeaker();

                // 🔁 این بخش را همین‌جا اضافه یا اصلاح کن
                for (const [userId, user] of Object.entries(users.value)) {
                    if (userId === id) continue;

                    const peer = createPeer(userId);
                    peers[userId] = peer;

                    localStream.value.getTracks().forEach(track => {
                        peer.addTrack(track, localStream.value);
                    });

                    const offer = await peer.createOffer();
                    await peer.setLocalDescription(offer);

                    socket.emit('offer', {
                        target: user.socketId,
                        sdp: offer,
                        caller: id
                    });
                }
            }

            if (role.value === 'listener') {
                socket.emit('listener-joined', { room: roomId, listenerId: myId.value });
            }
        });


        socket.on('resend-offer', async (listenerId) => {
            const user = Object.entries(users.value).find(([id, u]) => u.socketId === listenerId);
            if (!user) return;

            const [userId, userInfo] = user;

            const peer = createPeer(userId);
            peers[userId] = peer;

            localStream.value.getTracks().forEach(track => {
                peer.addTrack(track, localStream.value);
            });

            const offer = await peer.createOffer();
            await peer.setLocalDescription(offer);

            socket.emit('offer', {
                target: userInfo.socketId,
                sdp: offer,
                caller: myId.value
            });
        });

        socket.on('send-offer-to-listener', async ({ listenerId }) => {
            const user = Object.entries(users.value).find(([id, u]) => u.socketId === listenerId);
            if (!user) return;

            const [userId, userInfo] = user;

            const peer = createPeer(userId);
            peers[userId] = peer;

            localStream.value.getTracks().forEach(track => {
                peer.addTrack(track, localStream.value);
            });

            const offer = await peer.createOffer();
            await peer.setLocalDescription(offer);

            socket.emit('offer', {
                target: userInfo.socketId,
                sdp: offer,
                caller: myId.value
            });
        });

        socket.on('reaction', ({ userId, emoji }) => {
            const el = document.querySelector(`#reaction-${userId}`);
            if (el) {
                el.innerText = emoji;
                el.classList.add('show');

                setTimeout(() => {
                    el.classList.remove('show');
                    el.innerText = '';
                }, 3000);
            } else {
                console.warn('❌ Reaction element not found for userId:', userId);
            }
        });


        socket.on('room-cleared', () => {
            for (const peer of Object.values(peers)) {
                peer.onicecandidate = null;
                peer.ontrack = null;
                peer.close();
            }

            if (localStream.value) {
                localStream.value.getTracks().forEach(track => track.stop());
                localStream.value = null;
            }

            Object.keys(peers).forEach(id => delete peers[id]);

            document.querySelectorAll('audio.hidden-audio').forEach(audio => {
                audio.pause();
                audio.srcObject = null;
                audio.remove();
            });

            // 👉 آلارم بزن به صفحه اصلی که باید خارج شه
            window.dispatchEvent(new CustomEvent('room-cleared-by-host'));

            socket.disconnect();
        });
    };

    const raiseHand = () => {
        socket.emit('raise-hand', roomId);
    };

    const changeMicStatus = (micEnabled) => {
        if (localStream.value) {
            localStream.value.getAudioTracks().forEach(track => {
                track.enabled = !micEnabled;
            });
        }
    };

    const clearRoom = () => {
        socket.emit('clear-room', roomId);

        document.querySelectorAll('audio.hidden-audio').forEach(audio => {
            audio.pause();
            audio.srcObject = null;
            audio.remove();
        });
    };

    const leaveRoom = () => {
        socket.emit('leave-room', roomId);

        for (const peer of Object.values(peers)) {
            peer.onicecandidate = null;
            peer.ontrack = null;
            peer.close();
        }

        if (localStream.value) {
            localStream.value.getTracks().forEach(track => track.stop());
            localStream.value = null;
        }

        Object.keys(peers).forEach(id => delete peers[id]);

        document.querySelectorAll('audio.hidden-audio').forEach(audio => {
            audio.pause();
            audio.srcObject = null;
            audio.remove();
        });

        socket.disconnect();
    };

    const muteListener = () => {
        if (roleItem === 'host') {
            socket.emit('force-listeners', roomId);
        }
    };

    const sendReaction = (emoji) => {
        socket.emit('send-reaction', { room: roomId, emoji });
    };


    return {
        joinRoom,
        raiseHand,
        leaveRoom,
        role,
        users,
        clearRoom,
        muteListener,
        sendReaction,
        changeMicStatus,
        localStream
    };
};
